<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 19.12.13 10:59
 */

namespace VM\FinancialStatementsBundle\Services\DataStructure;


class FinansowanieInwestycjiDS extends AbstractDataStructure implements InterfaceDataStructure
{
    /**
     * @var float
     */
    public $naklady = 0.00;

    /**
     * @var float
     */
    public $procentAmortyzacji = 0.00;

    /**
     * @var int
     */
    public $rokPoczatkowy = 2013;

    /**
     * @var int
     */
    public $liczbaLat = 5;

    /**
     * @var bool
     */
    public $czyKapitalWlasny = false;

    /**
     * @var bool
     */
    public $czyKredyt = false;

    /**
     * @var bool
     */
    public $czyLeasing = false;

    /**
     * @var bool
     */
    public $czyNaSrodkiTrwale = false;

    /**
     * @var float
     */
    public $kapitalWlasny = 0.00;

    /**
     * @var float
     */
    public $kapitalObcy = 0.00;

    /**
     * @var float
     */
    public $oprocentowanieKredytuLubLeasingu = 0.00;

    /**
     * @var int
     */
    public $okresKredytuLubLeasingu = 0;

    /**
     * @var float
     */
    public $ostatniaWplataLeasingu = 0.00;

    /**
     * @var int
     */
    public $iloscMiejscPoPrzecinku = 2;

    protected function validateNaklady()
    {
        if(!is_numeric($this->naklady))
        {
            throw new \Exception('Nakłady muszą być liczbą!');
        }

        if($this->naklady < 0)
        {
            throw new \Exception('Nakłady muszą być większe od 0!');
        }
    }

    protected function validateProcentAmortyzacji()
    {
        if(!is_numeric($this->procentAmortyzacji))
        {
            throw new \Exception('Błędny procent! Procent musi być liczbą!');
        }

        if($this->procentAmortyzacji > 1)
        {
            throw new \Exception('Błędny procent! Procent musi być mniejszy lub równy 1!');
        }

        if($this->procentAmortyzacji < 0)
        {
            throw new \Exception('Błędny procent! Procent musi być większy lub równy 0!');
        }
    }

    protected function validateRokPoczatkowy()
    {
        if(!is_numeric($this->rokPoczatkowy) || !is_int($this->rokPoczatkowy))
        {
            throw new \Exception('Rok początkowy musi być liczbą całkowitą!');
        }

        if($this->rokPoczatkowy < 0)
        {
            throw new \Exception('Rok początkowy musi być większy lub równy 0!');
        }
    }

    protected function validateLiczbaLat()
    {
        if(!is_numeric($this->liczbaLat) || !is_int($this->liczbaLat*1))
        {
            throw new \Exception('Liczba lat musi być liczbą całkowitą!');
        }

        if($this->liczbaLat < 1)
        {
            throw new \Exception('Liczba lat musi być większa od 1!');
        }
    }

    protected function validateKapitaly()
    {
        if(!is_numeric($this->kapitalWlasny))
        {
            throw new \Exception('Kapitał własny musi być liczbą rzeczywistą!');
        }

        if(!is_numeric($this->kapitalObcy))
        {
            throw new \Exception('Kapitał obcy musi być liczbą rzeczywistą!');
        }

        if($this->kapitalWlasny < 0)
        {
            throw new \Exception('Kapitał własny musi być większy lub równy 0!');
        }

        if($this->kapitalObcy < 0)
        {
            throw new \Exception('Kapitał obcy musi być większy lub równy 0!');
        }
    }

    protected function validateOprocentowanieKredytuLubLeasingu()
    {
        if(($this->czyLeasing || $this->czyKredyt))
        {
            if(!is_numeric($this->oprocentowanieKredytuLubLeasingu))
            {
                throw new \Exception('Oprocentowanie kredytu lub leasingu musi być liczbą rzeczywistą!');
            }

            if($this->oprocentowanieKredytuLubLeasingu < 0)
            {
                throw new \Exception('Oprocentowanie kredytu lub leasingu musi być większe lub równe 0!');
            }

            if($this->oprocentowanieKredytuLubLeasingu > 1)
            {
                throw new \Exception('Oprocentowanie kredytu lub leasingu musi być mniejsze lub równe 1!');
            }
        }
    }

    protected function validateOkresKredytuLubLeasingu()
    {
        if(($this->czyLeasing || $this->czyKredyt))
        {
            if(!is_numeric($this->okresKredytuLubLeasingu) || !is_int($this->okresKredytuLubLeasingu))
            {
                throw new \Exception('Okres kredytu lub leasingu musi być liczbą całkowitą!');
            }

            if($this->okresKredytuLubLeasingu < 1)
            {
                throw new \Exception('Okres kredytu lub leasingu musi być większy lub równy 1!');
            }
        }
    }

    protected function validateOstatniaWplataLeasingu()
    {
        if($this->czyLeasing)
        {
            if(!is_numeric($this->okresKredytuLubLeasingu) || !is_int($this->okresKredytuLubLeasingu))
            {
                throw new \Exception('Ostatnia wpłata leasingu musi być liczbą całkowitą!');
            }

            if($this->okresKredytuLubLeasingu < 0)
            {
                throw new \Exception('Ostatnia wpłata leasingu musi być większa lub równa 0!');
            }
        }
    }

    public function validate()
    {
        $this->validateNaklady();
        $this->validateProcentAmortyzacji();
        $this->validateRokPoczatkowy();
        $this->validateLiczbaLat();

        if(is_bool($this->czyKapitalWlasny) === false)
        {
            throw new \Exception('Niepoprawne oznacznie czy kapitał własny. True/false.');
        }

        if(is_bool($this->czyKredyt) === false)
        {
            throw new \Exception('Niepoprawne oznacznie czy kredyt. True/false.');
        }

        if(is_bool($this->czyLeasing) === false)
        {
            throw new \Exception('Niepoprawne oznacznie czy leasing. True/false.');
        }

        if(is_bool($this->czyNaSrodkiTrwale) === false)
        {
            throw new \Exception('Niepoprawne oznacznie czy na środki trwałe. True/false.');
        }

        if((($this->czyKapitalWlasny || $this->czyKredyt) xor $this->czyLeasing) == 0)
        {
            throw new \Exception('Kapitał własny i/lub kredyt nie mogą występować wraz z leasingiem.');
        }

        $this->validateKapitaly();

        if(!$this->czyLeasing && ($this->kapitalWlasny + $this->kapitalObcy) != $this->naklady)
        {
            throw new \Exception('Suma kapitału własnego i obcego nie jest równa nakładom.');
        }

        $this->validateOprocentowanieKredytuLubLeasingu();
        $this->validateOkresKredytuLubLeasingu();
        $this->validateOstatniaWplataLeasingu();

        return true;
    }
} 